<?php

namespace App\Http\Controllers\Api;

use App\Models\CustomerModel;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Auth;




class CustomerAuthController extends Controller
{
    /**
     * Register a new customer and return a JWT token.
     */
    public function register(Request $request)
    {
        // Define validation rules
        $rules = [
            'name'     => 'required|string|max:255',
            'email'    => 'required|string|email|max:255|unique:customers',
            'phone'    => 'required|string|max:20|unique:customers,mob',
            'password' => 'required|string|min:6|confirmed',
        ];

        // Define custom messages for validation errors
        $messages = [
            'name.required'       => 'The name field is required.',
            'email.required'      => 'The email field is required.',
            'email.email'         => 'Please enter a valid email address.',
            'email.unique'        => 'The email address is already in use.',
            'phone.required'      => 'The phone field is required.',
            'phone.unique'        => 'The phone number is already in use. Please choose a different phone number.',
            'password.required'   => 'The password field is required.',
            'password.min'        => 'The password must be at least 6 characters.',
            'password.confirmed'  => 'The password confirmation does not match.',
        ];

        // Perform the validation
        $validator = Validator::make($request->all(), $rules, $messages);

        // If validation fails, return a custom JSON response
        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => $validator->errors()->first()
            ], 422);
        }

        // Create the customer record
        $customer = CustomerModel::create([
            'name'     => $request->name,
            'email'    => $request->email,
            'role'     => '2', // Assuming "2" signifies a customer role
            'mob'      => $request->phone,
            'password' => Hash::make($request->password),
            'pass'     => $request->password, // If you need to store the plain password; otherwise, remove it.
        ]);

        // Log in the customer and generate a JWT token
        $token = auth('customer')->login($customer);

        // Return the success response with token details
        return response()->json([
            'status'  => 'success',
            'message' => 'Registration successful.',
            'data'    => [
                'access_token' => $token,
                'token_type'   => 'bearer',
                'expires_in'   => auth('customer')->factory()->getTTL() * 60,
            ]
        ], 200);
    }


    /**
     * Authenticate an existing customer and return a JWT token.
     */
    public function login(Request $request)
    {
     // Extract credentials from request.
     $credentials = $request->only('mob', 'password');

     // Attempt login on the 'customer' guard.
     if (!$token = auth('customer')->attempt($credentials)) {
         return response()->json([
             'status'  => 'error',
             'message' => 'Invalid Phone Number or password',
             'data'    => []
         ], 401);
     }
 
     // Return the custom JSON response with the JWT token details.
     return response()->json([
         'status'  => 'success',
         'message' => 'Login successful.',
         'data'    => [
             'access_token' => $token,
             'token_type'   => 'bearer',
             'expires_in'   => auth('customer')->factory()->getTTL() * 60,
         ]
     ], 200);
    }


    public function profile()
    {
        try {
            $user = auth('customer')->user();
    
            // Check if a user is authenticated
            if (!$user) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'Session expired or you are logged out. Please log in again.',
                    'data'    => []
                ], 401);
            }
    
            // Return the customer data in your custom format
            return response()->json([
                'status'  => 'success',
                'message' => 'User profile retrieved successfully.',
                'data'    => $user
            ], 200);
    
        } catch (\Illuminate\Auth\AuthenticationException $e) {
            return response()->json([
                'status'  => 'error',
                'message' => 'You are logged out. Kindly Relogin.',
                'data'    => []
            ], 401);
        } catch (\Tymon\JWTAuth\Exceptions\TokenInvalidException $e) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Invalid token. Please log in again.',
                'data'    => []
            ], 401);
        } catch (\Tymon\JWTAuth\Exceptions\TokenExpiredException $e) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Token has expired. Please log in again.',
                'data'    => []
            ], 401);
        } catch (\Exception $e) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Something went wrong. Please try again later.',
                'data'    => []
            ], 500);
        }
    }

    /**
     * Logout the customer (invalidate the token).
     */
    public function logout()
    {
        auth('customer')->logout();

        // Return a custom JSON response
        return response()->json([
            'status'  => 'success',
            'message' => 'Logged out successfully.',
            'data'    => []
        ], 200);
    }

    /**
     * Refresh and return a new JWT token.
     */
    public function refresh()
    {
        return $this->respondWithToken(auth('customer')->refresh());
    }

    /**
     * Format the token response in a standard JSON structure.
     */
    protected function respondWithToken($token)
    {
        return response()->json([
            'access_token' => $token,
            'token_type'   => 'bearer',
            'expires_in'   => auth('customer')->factory()->getTTL() * 60, // TTL in seconds
        ]);
    }


    public function sendOtp(Request $request)
    {


        $phoneno=$request->phone_no;
        if (!$phoneno) {
            return response()->json(['status' => 'error', 'message' => 'phone number is Required.'], 404);
        }

     

        $customer = CustomerModel::where('mob', $request->phone_no)->first();
        if (!$customer) {
            return response()->json(['status' => 'error', 'message' => 'No user exists with this phone number.'], 404);
        }

        $otp = rand(100000, 999999);
        Cache::put('otp_' . $request->phone_no, $otp, now()->addMinutes(10));
        $apikey='676933437ed08';
        $senderid='MAGCON';

        $mobile = $request->phone_no;
        $message = urlencode("$otp is your One Time Verification(OTP) to confirm your phone no. at MagnetConnects Venture Private Limited");
        $apiUrl = "https://www.smsalert.co.in/api/push.json?apikey=$apikey&sender=$senderid&mobileno=$mobile&text=$message";
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $apiUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode == 200) {
            return response()->json(['status' => 'success', 'message' => 'OTP sent successfully.', 'otp' => $otp,], 200);
        } else {
            return response()->json(['status' => 'error', 'message' => 'Failed to send OTP. Please check your API credentials.'], 500);
        }
    }

    public function loginWithOtp(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone_no' => 'required|digits:10',
            'otp' => 'required|digits:6'
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => 'error', 'message' => $validator->errors()->first()], 400);
        }

        $cachedOtp = Cache::get('otp_' . $request->phone_no);

        if (!$cachedOtp || $cachedOtp != $request->otp) {
            return response()->json(['status' => 'error', 'message' => 'Invalid or expired OTP.'], 401);
        }

    
        $customerlog = CustomerModel::where('mob', $request->phone_no)->first();
        if (!$customerlog) {
            return response()->json(['status' => 'error', 'message' => 'Customer not found.'], 404);
        }

 
        $token = auth('customer')->login($customerlog);
        Cache::forget('otp_' . $request->phone_no);

        return response()->json([
            'status' => 'success',
            'message' => 'Login successful.',
            'data' => [
                'access_token' => $token,
                'token_type' => 'bearer',
                'expires_in' => auth('customer')->factory()->getTTL() * 60,
            ]
        ], 200);
    }
}
